import sys
import re
import numpy as np
from numpy import sqrt, exp, sin, cos, tan, pi
import numpy.linalg as LA 
import csv
from matplotlib import pyplot as plt


"""
TFTのIds-Vgs-Vdsデータから飽和移動度を求める
"""

#===================================
# physical constants
#===================================
pi   = 3.14159265358979323846
pi2  = 2.0 * pi
h    = 6.6260755e-34    # Js";
hbar = 1.05459e-34      # "Js";
c    = 2.99792458e8     # m/s";
e    = 1.60218e-19      # C";
e0   = 8.854418782e-12; # C<sup>2</sup>N<sup>-1</sup>m<sup>-2</sup>";
kB   = 1.380658e-23     # JK<sup>-1</sup>";
me   = 9.1093897e-31    # kg";
R    = 8.314462618      # J/K/mol
a0   = 5.29177e-11      # m";


#===================================
# parameters
#===================================
infile = 'TransferCurve.csv'

dg  = 100.0e-9 # m
erg = 11.9

W = 300.0e-6 # m
L =  50.0e-6

Vds0 = 10.0    # V
xfitmin = 1.90 # V
xfitmax = 10.0

#===================================
# figure configuration
#===================================
fontsize        = 12
legend_fontsize = 6


#=============================
# other functions
#=============================
# 余計な文字が含まれている文字列から、
# 浮動小数点に変換できる最初の文字列を切り出し、
# 浮動小数点に変換して返す
def pfloat(str, defval = None):
# 文字列から、浮動小数点に使える文字が連続している部分を切り出す
    m = re.search(r'([+\-eE\d\.]+)', str)
#    print("str, m=", str, m)
# 一致した文字列を取得
    valstr = m.group()
    try:
        return float(valstr)
    except:
        return defval

def savecsv(outfile, header, datalist):
    try: 
        print("Write to [{}]".format(outfile))
        f = open(outfile, 'w')
    except:
#    except IOError:
        print("Error: Can not write to [{}]".format(outfile))
    else:
        fout = csv.writer(f, lineterminator='\n')
        fout.writerow(header)
#        fout.writerows(data)
        for i in range(0, len(datalist[0])):
            a = []
            for j in range(len(datalist)):
                a.append(datalist[j][i])
            fout.writerow(a)
        f.close()

def read_csv(fname):
    print("")
    with open(fname) as f:
        fin = csv.reader(f)
        
        labels = next(fin)
        xlabel = labels[0]

# label行が 空文字 の場合、データとしては読み込まない
        ylabels = []
        for i in range(1, len(labels)):
            if labels[i] == '':
                break
            ylabels.append(labels[i])
        ny = len(ylabels)
#        print("xlabel: ", xlabel)
#        print("ylabels: ", ylabels)
#        print("ny=", ny)

        x     = []
        ylist = []
        for i in range(ny):
            ylist.append([])
        
        for row in fin:
            x.append(pfloat(row[0]))
            for i in range(1, ny+1):
                v = pfloat(row[i])
                if v is not None:
                    ylist[i-1].append(v)
                else:
                    ylist[i-1].append(None)

    return xlabel, ylabels, x, ylist


def main():
    Cox = erg * e0 / dg  #(F/m^2)
    print("")
    print("Cox = {:12.6g} [F/m^2]".format(Cox))

    xlabel, ylabels, Vgs, IdsVgs = read_csv(infile)
    nVgs = len(IdsVgs[0])
    nVds = len(ylabels)
    Vds = []
    for i in range(nVds):
        Vds.append(pfloat(ylabels[i]))
    print("")
    print("nVds=", nVds)
    print("nVgs=", nVgs)
    print("xlabel : ", xlabel)
    print("ylabels: ", ylabels)
    print("Vds: ", Vds)
#    for i in range(len(Idslist)):
#        print("")
#        print("Idslist[{}]: {}".format(i, Idslist[i]))

# 出力特性 Ids - Vds をプロットするためのデータを作る
    IdsVds = np.empty([nVgs, nVds])
    for ig in range(nVgs):
        for id in range(nVds):
            IdsVds[ig][id] = IdsVgs[id][ig]

# sqrt(Ids) - Vgsプロットをする Vds0 のデータ番号 iVds を探す
# Vds[i] が小さい方から順に、Vds0 <= Vds[i] となる i を探すが、
# 浮動小数点誤差があることを考慮し、Vds0 - 1.0e-3 <= Vds[i] とする
    for i in range(nVds):
        if Vds0 - 1.0e-3 <= Vds[i]:
            iVds = i
            break
    
# sqrt(Ids) データを作る
    print("")
    print("Vds used: {} V (iVds = {})".format(Vds[iVds], iVds))
    sqrtIds = []
    for ig in range(nVgs):
        sqrtIds.append(sqrt(IdsVgs[iVds][ig]))
#    print("sqrtIds=",sqrtIds)

# 最小二乗法のデータ
    xfit = []
    yfit = []
    for i in range(nVgs):
        if xfitmin <= Vgs[i] <= xfitmax:
            xfit.append(Vgs[i])
            yfit.append(sqrtIds[i])
    print("")
    print("Least squares fitting:")
    print("Vgs range: {} - {} V".format(xfitmin, xfitmax))
    print("Vgs=", xfit)
    print("Igs^(1/2)=", yfit)
    ai = np.polyfit(xfit, yfit, 1)            
# y = ai[1] + ai[0]x
# Vth = ai[1] / ai[0]
    Vth  = -ai[1] / ai[0]
    grad = ai[0]
    mu = grad * grad / (W * Cox / 2.0 / L)
    print("Vth = {:6.4g} V".format(Vth))
    print("dIgs^1/2/dVgs = {:12.4g} A^(1/2)/V".format(grad))
    print("mu_sat = {} m^2/Vs = {} cm^2/Vs".format(mu, 1.0e4 * mu))
    
    xcal = []
    ycal = []
    xx = xfitmin - 0.5
    xcal.append(xx)
    ycal.append(ai[1] + ai[0] * xx)
    xx = max(Vgs)
    xcal.append(xx)
    ycal.append(ai[1] + ai[0] * xx)

    print("")
    print("plot")
    fig = plt.figure(figsize = (12, 4))
    ax1 = fig.add_subplot(1, 3, 1)
    ax2 = fig.add_subplot(1, 3, 2)
    ax3 = fig.add_subplot(1, 3, 3)

# 伝達特性  Ids-Vgs のグラフ
    for id in range(nVds):
        ax1.plot(Vgs, IdsVgs[id],  linewidth = 0.5, marker = 'o', markersize = 0.5,
                    label = 'Vds={} V'.format(Vds[id]))
    for id in range(nVds):
        ax1.plot(Vgs, IdsVgs[id],  linewidth = 0.5, marker = 'o', markersize = 0.5,
                    label = 'Vds={} V'.format(Vds[id]))
    ax1.set_xlabel('Vgs (V)')
    ax1.set_ylabel("Ids (A)")
    ax1.set_yscale('log')
#    ax1.set_xlim([-0.5, 0.5])
#    ax1.set_ylim([0.0, 0.5])
    ax1.legend(loc = 'upper left', fontsize = legend_fontsize)

# 出力特性  Ids-Vds のグラフ
# 20点の Vgs を選び、そのうち、線形でIdsが見えるデータのみ表示する
    nskip = int(nVgs / 20.0 + 1.0e-6)
    for ig in range(0, nVgs, nskip):
# 線形プロットで見えないほどIdsが小さいデータは表示しない
        if max(IdsVds[ig]) < 1.0e-7:
            continue
        ax2.plot(Vds, IdsVds[ig],  linewidth = 0.5, marker = 'o', markersize = 0.5,
                    label = 'Vgs={} V'.format(Vgs[ig]))
    ax2.set_xlabel('Vds (V)')
    ax2.set_ylabel("Ids (A)")
#    ax2.set_xlim([-0.5, 0.5])
#    ax2.set_ylim([0.0, 0.5])
    ax2.legend(loc = 'upper left', fontsize = legend_fontsize)

# Ids^(1/2) - Vgs グラフ
    ax3.plot(Vgs, sqrtIds, linestyle = 'none', marker = 'o', markersize = 0.5)
    ax3.plot(xcal, ycal,   linestyle = '-', linewidth = 0.5)
    ax3.set_xlabel('Vgs (V)')
    ax3.set_ylabel("Ids$^{1/2}$ (A$^{1/2}$)")
#    ax2.set_xlim([-0.5, 0.5])
#    ax2.set_ylim([0.0, 0.5])
#    ax3.legend(loc = 'upper left', fontsize = legend_fontsize)

    plt.tight_layout()

    plt.pause(0.1)
    print("Press ENTER to exit>>", end = '')
    input()


if (__name__ == '__main__'):
    main()
